unit MotionBlending;

interface

uses
  CgTypes, CgGeometry, CgUtils, Math, OpenGL, MyGLInit, Dialogs,
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls,
  StdCtrls;

type
  pole = array of char;

  TCell = class
    cost:integer;
    direction:pole;
    constructor Create(c:integer);
    destructor Destroy();override;
  end;

  TQuaternion = class
    w,x,y,z:single;
    constructor Create(a,b,c,d:single);
    destructor Destroy();override;
  end;

  TAxisAngle = class
    axis:TCGVector;
    angle:single;
    constructor Create(a,b,c,d:single);
    destructor Destroy();override;
  end;

  TPointCloud = class
    id_frame:word;
    jpf:byte;                   //joints per frame
    data:array of real;
    constructor Create(f:word);
    procedure display(rr,gg,bb:byte);overload;
    procedure display(rr,gg,bb:byte;m:TCGMatrix);overload; //only for matrix check
    destructor Destroy();override;
  end;

  TBlending = class
    threshold,area:real;
    riesenie:TCell;
    zac_ries_i,zac_ries_j,j_max,i_max:word;             //size of matrix
    PC:array of TPointCloud;      //array of point clouds
    DP:array of array of TCell;   //matrix for dynamic programming
    table:array of array of integer;
    constructor Create();
    function Weighted_Average(w,a:array of real):real;
    function Tau(w, x1, z1, x2, z2:array of real):real;
    function X_0(w, x1, x2, z2:array of real; angle:real):real;
    function Z_0(w, x2, z1, z2:array of real; angle:real):real;
    function CIDF(w :array of real; angle,x0,z0:real; pc0,pc1:TPointCloud):real;
    procedure Timewarp(k,l:word;smer:char;d,r:byte;cena:integer;cesta:pole);
/////////////////////////////////////////////////////////////
    function QuatMult(a,b:TQuaternion):TQuaternion;
    function AxAnToQuat(ax:TAxisAngle):TQuaternion;
    function QuatToMat(a:TQuaternion):TCGMatrix;
    function QuatToAxAn(a:TQuaternion):TAxisAngle;
    function SLERP(a,b:TQuaternion; t:single):TQuaternion;
    destructor Destroy();override;
  end;

const neigh_frames:byte=5;

implementation

{ TQuaternion }

constructor TQuaternion.Create(a, b, c, d: single);
begin
  w:=a;
  x:=b;
  y:=c;
  z:=d;
end;

destructor TQuaternion.Destroy;
begin

  inherited;
end;

{ TBlending }

function TBlending.AxAnToQuat(ax: TAxisAngle): TQuaternion;
var norm,s:single;
begin

  with Result do
  begin
    s:=sin(ax.Angle/2);
    w:=cos(ax.Angle/2);
    x:=ax.Axis.x*s;
    y:=ax.Axis.y*s;
    z:=ax.Axis.z*s;
    norm:=sqrt(x*x+y*y+z*z+w*w);
    w:=w/norm;
    x:=x/norm;
    y:=y/norm;
    z:=z/norm;
  end;

end;

function TBlending.CIDF(w :array of real; angle,x0,z0:real; pc0,pc1:TPointCloud): real;
var i:word;
    fi,v1,v2,v3:real;
    matrix:TCGMatrix;

begin
  Result:=0;
  fi :=DegToRad(angle);

  for i:=0 to high(w) do
  begin
    v1:=pc1.data[i*3]*cos(fi)+pc1.data[i*3+2]*sin(fi)+x0;
    v2:=pc1.data[i*3+1];
    v3:=pc1.data[i*3]*(-sin(fi))+pc1.data[i*3+2]*cos(fi)+z0;

    Result:=Result + w[i]*(sqr(pc0.data[i*3]-v1{pc1.data[i*3]})+sqr(pc0.data[i*3+1]-v2{pc1.data[i*3+1]})+sqr(pc0.data[i*3+2]-v3{pc1.data[i*3+2]}));
  end;

end;

constructor TBlending.Create;
begin
  setlength(PC,2);
end;

destructor TBlending.Destroy;
begin

  inherited;
end;

function TBlending.QuatMult(a, b: TQuaternion): TQuaternion;
var va,vb,v :TCGVector;
begin

  va:=cgVector(a.x,a.y,a.z);
  vb:=cgVector(b.x,b.y,b.z);
  with Result do
  begin
    w:=a.w*b.w - cgDotProduct(va,vb);
    v:=cgVecAdd(cgVecScalarMult(vb, a.w), cgVecAdd(cgVecScalarMult(va, b.w), cgCrossProduct(va, vb)));
    x:=v.x;
    y:=v.y;
    z:=v.z;
  end;

end;

function TBlending.QuatToAxAn(a: TQuaternion): TAxisAngle;
var s:single;
begin

  with a do
  begin
    s:=sqrt(x*x + y*y + z*z);
    if s<>0 then
    begin
      Result.Axis.x:=x/s;
      Result.Axis.y:=y/s;
      Result.Axis.z:=z/s;
      Result.Angle:=2*cgArcCos(w);
    end
    else begin
      Result.Axis.x:=1;
      Result.Axis.y:=0;
      Result.Axis.z:=0;
      Result.Angle:=0;
    end;
  end;

end;

function TBlending.QuatToMat(a: TQuaternion): TCGMatrix;
begin
  with a do
  begin
    Result[0,0] := 1 - 2*y*y - 2*z*z;
    Result[1,0] := 2*x*y - 2*w*z;
    Result[2,0] := 2*x*z + 2*w*y;
    Result[3,0] := 0;

    Result[0,1] := 2*x*y + 2*w*z;
    Result[1,1] := 1 - 2*x*x - 2*z*z;
    Result[2,1] := 2*y*z - 2*w*x;
    Result[3,1] := 0;

    Result[0,2] := 2*x*z - 2*w*y;
    Result[1,2] := 2*y*z + 2*w*x;
    Result[2,2] := 1 - 2*x*x - 2*y*y;
    Result[3,2] := 0;

    Result[0,3] := 0;
    Result[1,3] := 0;
    Result[2,3] := 0;
    Result[3,3] := 1;
  end;

end;

function TBlending.SLERP(a, b: TQuaternion; t: single): TQuaternion;
var
  epsilon, pom1, pom2, i, j, k:single;
  q:TQuaternion;
begin
  epsilon:=0.0001;

  i:=a.x*b.x + a.y*b.y + a.z*b.z + a.w*b.w;

  if i<0 then
  begin
    i:=-i;
    q:=TQuaternion.Create(-b.w,-b.x,-b.y,-b.z);
  end
  else
    q:=TQuaternion.Create(b.w,b.x,b.y,b.z);

    if (1-i)<=epsilon then
  begin
    //linear interpolation
    pom1:=1-t;
    pom2:=t;
  end
  else
  begin
    //SLERP.
    j:=cgArcCos(i);
    k:=sin(j);
    pom1:=sin((1 - t)*j)/k;
    pom2:=sin(t*j)/k;
  end;

  Result.w:=pom1*a.w+pom2*q.w;
  Result.x:=pom1*a.x+pom2*q.x;
  Result.y:=pom1*a.y+pom2*q.y;
  Result.z:=pom1*a.z+pom2*q.z;

  q.Destroy;
end;

function TBlending.tau(w, x1, z1, x2, z2: array of real): real;
var i:word;
    sum1,sum2,numerator,denominator:real;
begin
  sum1:=0;
  sum2:=0;
  for i:=0 to high(w) do
    sum1:=sum1+w[i]*(x1[i]*z2[i]-x2[i]*z1[i]);
  for i:=0 to high(w) do
    sum2:=sum2+w[i]*(x1[i]*x2[i]+z1[i]*z2[i]);

  numerator:=sum1 - (weighted_average(w,x1)*weighted_average(w,z2) - weighted_average(w,x2)*weighted_average(w,z1));
  denominator:=sum2 - (weighted_average(w,x1)*weighted_average(w,x2) + weighted_average(w,z1)*weighted_average(w,z2));

  Result:=RadToDeg(arctan(numerator/denominator));
end;

procedure TBlending.Timewarp(k,l:word;smer:char;d,r:byte;cena:integer;cesta:pole);
var i,int:integer;
begin
  if table[k,l] < area then
  begin

    int:=cena+table[k,l];
    setlength(cesta,length(cesta)+1);
    cesta[high(cesta)]:=smer;

    if (k < i_max) and (l < j_max) then
    begin
      if int < DP[k,l].cost then
      begin
        DP[k,l].cost:=int;
        setlength(DP[k,l].direction,length(cesta));
        DP[k,l].direction:=cesta;

{if d=0 then
  if table[k+1,l] < (table[k+1,l+1] - threshold) then
    Timewarp(k+1,l,'r',2,r-1,int,cesta)
  else
    Timewarp(k+1,l+1,'s',2,2,int,cesta)
else
if r=0 then
  if table[k,l+1] < (table[k+1,l+1] - threshold) then
    Timewarp(k,l+1,'d',d-1,2,int,cesta)
  else
    Timewarp(k+1,l+1,'s',2,2,int,cesta)
else
//if (d>0) and (r>0) then
  if table[k,l+1] <= table[k+1,l] then
    if table[k,l+1] < (table[k+1,l+1] - threshold) then
      Timewarp(k,l+1,'d',d-1,2,int,cesta)
    else
      Timewarp(k+1,l+1,'s',2,2,int,cesta)
  else
    if table[k+1,l] < (table[k+1,l+1] - threshold) then
      Timewarp(k+1,l,'r',2,r-1,int,cesta)
    else
      Timewarp(k+1,l+1,'s',2,2,int,cesta);
}
      if (d > 0) and (table[k,l+1] <= (table[k+1,l+1] )) then
        Timewarp(k,l+1,'d',d-1,2,int,cesta);

      if (r > 0) and (table[k+1,l] <= (table[k+1,l+1] )) then
        Timewarp(k+1,l,'r',2,r-1,int,cesta);

      Timewarp(k+1,l+1,'s',2,2,int,cesta);
    end;
  end
  else
  if riesenie.cost > cena then
  begin
    riesenie.cost:=cena;
    setlength(riesenie.direction,length(cesta));
    riesenie.direction:=cesta;
  end;
end;

end;

function TBlending.weighted_average(w, a: array of real): real;
var i:word;
begin
  if length(w) <> length(a) then
    Showmessage('Incorrect number of weights');
  Result:=0;
  for i:=0 to high(a) do
    Result:=Result+w[i]*a[i];
end;

function TBlending.X_0(w, x1, x2, z2: array of real; angle: real): real;
var i:integer;
    a,b,c:real;
begin
  a:=Weighted_Average(w,x1);
  b:=Weighted_Average(w,x2)*cos(DegToRad(angle));
  c:=Weighted_Average(w,z2)*sin(DegToRad(angle));

  Result:= a - b - c;
end;

function TBlending.Z_0(w, x2, z1, z2: array of real; angle: real): real;
var a,b,c:real;
begin
  a:=Weighted_Average(w,z1);
  b:=Weighted_Average(w,x2)*sin(DegToRad(angle));
  c:=Weighted_Average(w,z2)*cos(DegToRad(angle));

  Result:= a + b - c;
end;

{ TAxisAngle }

constructor TAxisAngle.Create(a, b, c, d: single);
begin
  axis.x:=a;
  axis.y:=b;
  axis.z:=c;
  angle:=d;
end;

destructor TAxisAngle.Destroy;
begin

  inherited;
end;

{ TPointCloud }

constructor TPointCloud.Create(f:word);
begin
  id_frame:=f;
  jpf:=0;
  setlength(data,0);
end;

destructor TPointCloud.Destroy;
begin

  inherited;
end;

procedure TPointCloud.display(rr,gg,bb:byte);
var i,j:word;
begin
  glPointSize(15);
  glBegin(GL_POINTS);
    for j:=0 to neigh_frames-1 do      //for each frame
    begin
        material_color(rr/255,gg/255,bb/255,1);
      for i:=0 to jpf-1 do             //for each joint
        glVertex3f(data[j*jpf*3+i*3],data[j*jpf*3+i*3+1],data[j*jpf*3+i*3+2]);
    end;
  glEnd();
end;

procedure TPointCloud.display(rr, gg, bb: byte; m: TCGMatrix);
var i,j:word;
    v1,v2,v3,v4:real;
begin
  glPointSize(15);
  glBegin(GL_POINTS);
    for j:=0 to neigh_frames-1 do      //for each frame
    begin
        material_color(rr/255,gg/255,bb/255,1);
      for i:=0 to jpf-1 do             //for each joint
      begin
        v1:=data[j*jpf*3+i*3]*m[0][0]+data[j*jpf*3+i*3+2]*m[2][0]+m[3][0];
        v2:=data[j*jpf*3+i*3+1];
        v3:=data[j*jpf*3+i*3]*m[0][2]+data[j*jpf*3+i*3+2]*m[2][2]+m[3][2];
        glVertex3f(v1,v2,v3);
      end
    end;
  glEnd();
end;

{ TCell }

constructor TCell.Create(c:integer);
begin
  cost:=c;
end;

destructor TCell.Destroy;
begin

  inherited;
end;

end.
